/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.examples.concurrency.foreach;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jbpm.api.ProcessInstance;
import org.jbpm.api.task.Task;
import org.jbpm.test.JbpmTestCase;

/**
 * @author Tom Baeyens
 */
public class ForEachTest extends JbpmTestCase {

  String deploymentId;
  String deptSales;
  String deptHR;
  String deptFinance;

  protected void setUp() throws Exception {
    super.setUp();

    // create identities
    deptSales = identityService.createGroup("sales-dept");
    deptHR = identityService.createGroup("hr-dept");
    deptFinance = identityService.createGroup("finance-dept");

    identityService.createUser("johndoe", "John", "Doe");
    identityService.createMembership("johndoe", deptSales, "SalesManager");

    identityService.createUser("joesmoe", "Joe", "Smoe");
    identityService.createMembership("joesmoe", deptHR, "HRManager");

    identityService.createUser("janedoe", "Jane", "Doe");
    identityService.createMembership("janedoe", deptFinance, "FinanceManager");

    deploymentId = repositoryService.createDeployment().addResourceFromClasspath("org/jbpm/examples/concurrency/foreach/process.jpdl.xml").deploy();
  }

  protected void tearDown() throws Exception {
    repositoryService.deleteDeploymentCascade(deploymentId);

    // delete identities
    identityService.deleteGroup(deptSales);
    identityService.deleteGroup(deptHR);
    identityService.deleteGroup(deptFinance);
    identityService.deleteUser("johndoe");
    identityService.deleteUser("joesmoe");
    identityService.deleteUser("janedoe");

    super.tearDown();
  }

  public void testForEachCompleteAll() {
    Map<String, Object> variables = new HashMap<String, Object>();
    variables.put("departments", new String[] { "sales-dept", "hr-dept", "finance-dept" });
    variables.put("quorum", 3);

    ProcessInstance processInstance = executionService.startProcessInstanceByKey("ForEach", variables);
    // there should be 3 forked executions - same as number of departments
    assertEquals(3, processInstance.getExecutions().size());

    List<Task> taskListSales = taskService.findGroupTasks("johndoe");
    assertEquals("Expected a single task in johndoe's task list", 1, taskListSales.size());

    List<Task> taskListHR = taskService.findGroupTasks("joesmoe");
    assertEquals("Expected a single task in joesmoe's task list", 1, taskListHR.size());

    List<Task> taskListFinance = taskService.findGroupTasks("janedoe");
    assertEquals("Expected a single task in janedoe's task list", 1, taskListFinance.size());

    // a member of sales department takes the task
    taskService.takeTask(taskListSales.get(0).getId(), "johndoe");

    taskListSales = taskService.findPersonalTasks("johndoe");
    assertEquals("Expected a single task being created", 1, taskListSales.size());
    // complete collect data from sales department
    taskService.completeTask(taskListSales.get(0).getId());

    // next a member of HR department takes the task
    taskService.takeTask(taskListHR.get(0).getId(), "joesmoe");

    taskListHR = taskService.findPersonalTasks("joesmoe");
    assertEquals("Expected a single task being created", 1, taskListHR.size());
    // complete collect data from HR department
    taskService.completeTask(taskListHR.get(0).getId());
    
    // finally a member of Finance department takes the task
    taskService.takeTask(taskListFinance.get(0).getId(), "janedoe");

    taskListFinance = taskService.findPersonalTasks("janedoe");
    assertEquals("Expected a single task being created", 1, taskListFinance.size());
    // complete collect data from HR department
    taskService.completeTask(taskListFinance.get(0).getId());

    Date endTime = historyService.createHistoryProcessInstanceQuery().processInstanceId(processInstance.getId()).uniqueResult().getEndTime();
    assertNotNull(endTime);
  }

  public void testForEachCompleteAfterTwoJoined() {
    Map<String, Object> variables = new HashMap<String, Object>();
    variables.put("departments", new String[] { "sales-dept", "hr-dept", "finance-dept" });
    variables.put("quorum", 2);
    ProcessInstance processInstance = executionService.startProcessInstanceByKey("ForEach", variables);

    // there should be 3 forked executions - same as number of departments
    assertEquals(3, processInstance.getExecutions().size());

    List<Task> taskListSales = taskService.findGroupTasks("johndoe");
    assertEquals("Expected a single task in johndoe's task list", 1, taskListSales.size());

    List<Task> taskListHR = taskService.findGroupTasks("joesmoe");
    assertEquals("Expected a single task in joesmoe's task list", 1, taskListHR.size());

    List<Task> taskListFinance = taskService.findGroupTasks("janedoe");
    assertEquals("Expected a single task in janedoe's task list", 1, taskListFinance.size());

    // a member of sales department takes the task
    taskService.takeTask(taskListSales.get(0).getId(), "johndoe");

    taskListSales = taskService.findPersonalTasks("johndoe");
    assertEquals("Expected a single task being created", 1, taskListSales.size());
    // complete collect data from sales department
    taskService.completeTask(taskListSales.get(0).getId());

    // next a member of HR department takes the task
    taskService.takeTask(taskListHR.get(0).getId(), "joesmoe");

    taskListSales = taskService.findPersonalTasks("joesmoe");
    assertEquals("Expected a single task being created", 1, taskListSales.size());
    // complete collect data from HR department
    taskService.completeTask(taskListSales.get(0).getId());

    Date endTime = historyService.createHistoryProcessInstanceQuery().processInstanceId(processInstance.getId()).uniqueResult().getEndTime();
    assertNotNull(endTime);
  }
}
